import React, { useState, useRef, useEffect, useCallback, useLayoutEffect } from 'react';
import { Trash2, CheckCircle, AlertCircle, Clock, Plus, MoreVertical, Maximize2 } from 'lucide-react';
import { parseCSV } from '../lib/csv';

// Column Helper
export const COLS = [
    { key: 'select', width: 40, label: '', isAction: true },
    { key: 'keyword', width: 250, label: 'Keyword', isEditable: true },
    { key: 'secondaryKeyword', width: 200, label: 'Secondary', isEditable: true },
    { key: 'nlpOutline', width: 200, label: 'Outline', isEditable: true },
    { key: 'intro', width: 200, label: 'Intro', isResult: true },
    { key: 'content', width: 200, label: 'Content', isResult: true },
    { key: 'faqs', width: 150, label: 'FAQs', isResult: true },
    { key: 'meta', width: 150, label: 'Meta', isResult: true },
    { key: 'title', width: 150, label: 'Title', isResult: true },
    { key: 'actions', width: 60, label: '', isAction: true }
];

export function DataTable({ tasks, selectedTasks, onToggleSelect, onUpdateTask, onBulkUpdateTask, onDeleteTask, onAddRow }) {
    const [colWidths, setColWidths] = useState(() => COLS.reduce((acc, col) => ({ ...acc, [col.key]: col.width }), {}));
    const [rowHeights, setRowHeights] = useState({});
    const [selStart, setSelStart] = useState(null);
    const [selEnd, setSelEnd] = useState(null);
    const [activeCell, setActiveCell] = useState(null); // { r, c }
    const [isDragging, setIsDragging] = useState(false);

    // Editor State
    const [editorState, setEditorState] = useState(null); // { taskId, field, rect, initialValue }

    const [contextMenu, setContextMenu] = useState(null);

    const containerRef = useRef(null);
    const dragRef = useRef(null);

    const handleMouseDown = (e, rIndex, cIndex) => {
        if (e.target.tagName === 'INPUT' && e.target.type === 'checkbox') return;
        if (cIndex === 9) return;
        if (e.button === 2) return;

        setSelStart({ r: rIndex, c: cIndex });
        setSelEnd({ r: rIndex, c: cIndex });
        setActiveCell({ r: rIndex, c: cIndex });
        setIsDragging(true);
        setEditorState(null); // Close editor on new selection
        setContextMenu(null);
    };

    const handleMouseEnter = (rIndex, cIndex) => {
        if (isDragging && selStart) {
            setSelEnd({ r: rIndex, c: cIndex });
        }
    };

    const handleMouseUp = () => {
        setIsDragging(false);
    };

    const handleRowContextMenu = (e, rIndex, taskId) => {
        e.preventDefault();
        setContextMenu({ x: e.clientX, y: e.clientY, rowIndex: rIndex, taskId: taskId });
        if (!selectedTasks.includes(taskId)) {
            onToggleSelect(taskId, true);
        }
    };

    useEffect(() => {
        window.addEventListener('mouseup', handleMouseUp);
        const closeMenu = () => setContextMenu(null);
        window.addEventListener('click', closeMenu);

        // Close editor on scroll to prevent misalignment
        const handleScroll = () => {
            if (editorState) setEditorState(null);
        };
        const container = containerRef.current;
        if (container) container.addEventListener('scroll', handleScroll);

        return () => {
            window.removeEventListener('mouseup', handleMouseUp);
            window.removeEventListener('click', closeMenu);
            if (container) container.removeEventListener('scroll', handleScroll);
        };
    }, [editorState]);

    const isSelected = (r, c) => {
        if (!selStart || !selEnd) return false;
        const minR = Math.min(selStart.r, selEnd.r);
        const maxR = Math.max(selStart.r, selEnd.r);
        const minC = Math.min(selStart.c, selEnd.c);
        const maxC = Math.max(selStart.c, selEnd.c);
        return r >= minR && r <= maxR && c >= minC && c <= maxC;
    };

    const getSelectionBounds = () => {
        if (!selStart || !selEnd) return null;
        return {
            minR: Math.min(selStart.r, selEnd.r),
            maxR: Math.max(selStart.r, selEnd.r),
            minC: Math.min(selStart.c, selEnd.c),
            maxC: Math.max(selStart.c, selEnd.c)
        };
    };

    const handleKeyDown = (e) => {
        if (editorState) return;

        if (e.ctrlKey && e.key === 'c') { handleCopy(e); return; }
        if (e.ctrlKey && e.key === 'v') { handlePaste(e); return; }
        if (e.key === 'Delete' || e.key === 'Backspace') { handleDeleteContent(e); }
    };

    const handleDeleteContent = (e) => {
        const bounds = getSelectionBounds();
        if (!bounds) return;
        const updates = [];
        for (let r = bounds.minR; r <= bounds.maxR; r++) {
            const task = tasks[r];
            if (!task) continue;
            const changes = {};
            let hasChanges = false;
            for (let c = bounds.minC; c <= bounds.maxC; c++) {
                const col = COLS[c];
                if (col.isEditable) {
                    changes[col.key] = "";
                    hasChanges = true;
                } else if (col.isResult) {
                    const currentRes = task.results || {};
                    const resKey = col.key === 'meta' ? 'meta' : col.key;
                    if (currentRes[resKey]) {
                        if (!changes.results) changes.results = { ...currentRes };
                        changes.results[resKey] = "";
                        hasChanges = true;
                    }
                }
            }
            if (hasChanges) updates.push({ id: task.id, changes });
        }
        if (updates.length > 0) onBulkUpdateTask(updates);
    };

    const handleBulkDeleteRows = () => {
        if (selectedTasks.length === 0 && contextMenu) {
            onDeleteTask(contextMenu.taskId);
        } else {
            selectedTasks.forEach(id => onDeleteTask(id));
        }
        setContextMenu(null);
    };

    const handleCopy = (e) => {
        const bounds = getSelectionBounds();
        if (!bounds) return;
        e.preventDefault();
        let text = "";
        for (let r = bounds.minR; r <= bounds.maxR; r++) {
            let rowText = [];
            for (let c = bounds.minC; c <= bounds.maxC; c++) {
                const task = tasks[r];
                if (!task) continue;
                const col = COLS[c];
                let val = "";
                if (col.isEditable) val = task[col.key] || "";
                else if (col.isResult) {
                    const keyMap = { 'intro': 'intro', 'content': 'content', 'faqs': 'faqs', 'meta': 'meta', 'title': 'title' };
                    val = task.results?.[keyMap[col.key]] || "";
                }
                rowText.push(val);
            }
            text += rowText.join("\t") + "\n";
        }
        navigator.clipboard.writeText(text);
    };

    const handlePaste = async (e) => {
        const bounds = getSelectionBounds();
        if (!bounds) return;
        e.preventDefault();

        const text = await navigator.clipboard.readText();
        const rows = parseCSV(text, '\t');

        if (rows.length === 0) return;

        const isSingleCell = rows.length === 1 && rows[0].length === 1;

        if (isSingleCell) {
            const rIndex = bounds.minR;
            const cIndex = bounds.minC;

            if (rIndex >= tasks.length) return;
            const task = tasks[rIndex];
            const col = COLS[cIndex];

            if (!col) return;

            const cellValue = rows[0][0];
            const changes = {};

            if (col.isEditable) {
                changes[col.key] = cellValue;
                onBulkUpdateTask([{ id: task.id, changes }]);
            } else if (col.isResult) {
                const resKey = col.key === 'meta' ? 'meta' : col.key;
                changes.results = { ...task.results, [resKey]: cellValue };
                onBulkUpdateTask([{ id: task.id, changes }]);
            }
            return;
        }

        let startR = bounds.minR;
        const updates = [];

        rows.forEach((cells, i) => {
            const rIndex = startR + i;
            if (rIndex >= tasks.length) return;
            const task = tasks[rIndex];
            let startC = bounds.minC;
            const changes = {};
            let hasChanges = false;

            cells.forEach((cellVal, j) => {
                const cIndex = startC + j;
                if (cIndex >= COLS.length) return;
                const col = COLS[cIndex];

                if (col.isEditable) {
                    changes[col.key] = cellVal;
                    hasChanges = true;
                }
                if (col.isResult) {
                    const resKey = col.key === 'meta' ? 'meta' : col.key;
                    if (!changes.results) changes.results = { ...task.results };
                    changes.results[resKey] = cellVal;
                    hasChanges = true;
                }
            });

            if (hasChanges) updates.push({ id: task.id, changes });
        });

        if (updates.length > 0) onBulkUpdateTask(updates);
    };

    const startColResize = (e, key) => {
        e.preventDefault();
        dragRef.current = { type: 'col', key, startX: e.pageX, startW: colWidths[key] };
        document.addEventListener('mousemove', handleResizeMove);
        document.addEventListener('mouseup', handleResizeUp);
    };
    const startRowResize = (e, msgId) => {
        e.preventDefault();
        const currentH = rowHeights[msgId] || 36;
        const isRowSelected = selectedTasks.includes(msgId);
        dragRef.current = { type: 'row', id: msgId, startY: e.pageY, startH: currentH, isRowSelected };
        document.addEventListener('mousemove', handleResizeMove);
        document.addEventListener('mouseup', handleResizeUp);
    };
    const handleResizeMove = (e) => {
        if (!dragRef.current) return;
        if (dragRef.current.type === 'col') {
            const diff = e.pageX - dragRef.current.startX;
            setColWidths(prev => ({ ...prev, [dragRef.current.key]: Math.max(40, dragRef.current.startW + diff) }));
        } else {
            const diff = e.pageY - dragRef.current.startY;
            const newH = Math.max(30, dragRef.current.startH + diff);
            setRowHeights(prev => {
                const next = { ...prev, [dragRef.current.id]: newH };
                if (dragRef.current.isRowSelected) {
                    selectedTasks.forEach(tid => { next[tid] = newH; });
                }
                return next;
            });
        }
    };
    const handleResizeUp = () => {
        dragRef.current = null;
        document.removeEventListener('mousemove', handleResizeMove);
        document.removeEventListener('mouseup', handleResizeUp);
    };

    const handleDoubleClick = (e, task, col) => {
        if (!col.isAction) {
            const rect = e.currentTarget.getBoundingClientRect();
            let val = "";
            if (col.isEditable) val = task[col.key] || "";
            else if (col.isResult) {
                val = task.results?.[col.key === 'meta' ? 'meta' : col.key] || "";
            }

            setEditorState({
                taskId: task.id,
                field: col.key,
                isResult: col.isResult,
                rect: {
                    top: rect.top,
                    left: rect.left,
                    width: rect.width,
                    height: rect.height
                },
                initialValue: val
            });
        }
    };

    const handleSaveEditor = (taskId, field, val, isResult) => {
        if (isResult) {
            const task = tasks.find(t => t.id === taskId);
            if (!task) return;
            const resKey = field === 'meta' ? 'meta' : field;
            const newRes = { ...task.results, [resKey]: val };
            onUpdateTask(taskId, { results: newRes });
        } else {
            onUpdateTask(taskId, { [field]: val });
        }
        setEditorState(null);
    };

    return (
        <div
            className="flex-1 flex flex-col h-full overflow-hidden bg-[var(--bg-tertiary)]"
            onKeyDown={handleKeyDown}
            tabIndex={0}
        >
            <div className="grid-wrapper" ref={containerRef} onContextMenu={(e) => e.preventDefault()}>
                <div className="header-row">
                    <div className="row-header" style={{ width: 40, minWidth: 40 }} onClick={() => onToggleSelect('all', true)}>#</div>
                    {COLS.map((col, cIndex) => (
                        <div key={col.key} className="header-cell" style={{ width: colWidths[col.key], minWidth: col.width }}>
                            {cIndex < 26 ? String.fromCharCode(65 + cIndex) : col.label}
                            <span className="ml-1">{col.label}</span>
                            <div className="col-resizer" onMouseDown={(e) => startColResize(e, col.key)}></div>
                        </div>
                    ))}
                </div>

                {tasks.map((task, rIndex) => (
                    <div key={task.id} className="data-row" style={{ height: rowHeights[task.id] || 36 }}>
                        <div
                            className={`row-header ${selectedTasks.includes(task.id) ? 'selected' : ''}`}
                            style={{ width: 40, minWidth: 40 }}
                            onClick={() => onToggleSelect(task.id, !selectedTasks.includes(task.id))}
                            onContextMenu={(e) => handleRowContextMenu(e, rIndex, task.id)}
                        >
                            {rIndex + 1}
                            <div className="row-resizer" onMouseDown={(e) => startRowResize(e, task.id)}></div>
                        </div>

                        {COLS.map((col, cIndex) => {
                            const isSel = isSelected(rIndex, cIndex);
                            const bounds = getSelectionBounds();
                            let borderClass = '';
                            if (isSel && bounds) {
                                if (rIndex === bounds.minR) borderClass += ' selected-top';
                                if (rIndex === bounds.maxR) borderClass += ' selected-bottom';
                                if (cIndex === bounds.minC) borderClass += ' selected-left';
                                if (cIndex === bounds.maxC) borderClass += ' selected-right';
                            }

                            let content = null;
                            if (col.key === 'select') {
                                content = <input type="checkbox" checked={selectedTasks.includes(task.id)} onChange={(e) => onToggleSelect(task.id, e.target.checked)} />;
                            } else if (col.key === 'actions') {
                                content = (
                                    <div className="flex gap-2 justify-center w-full">
                                        <Trash2 size={13} className="text-[var(--text-muted)] cursor-pointer hover:text-red-500" onClick={() => onDeleteTask(task.id)} />
                                    </div>
                                );
                            } else if (col.isResult) {
                                const val = task.results?.[col.key === 'meta' ? 'meta' : col.key];
                                content = (
                                    <div className="data-cell-content text-[var(--text-secondary)]">
                                        {val ? val.replace(/(<([^>]+)>)/gi, "").substring(0, 100) : ''}
                                    </div>
                                );
                            } else {
                                content = <div className="data-cell-content">{task[col.key]}</div>;
                            }

                            return (
                                <div
                                    key={col.key}
                                    className={`data-cell ${isSel ? 'selected ' + borderClass : ''} ${activeCell && activeCell.r === rIndex && activeCell.c === cIndex ? 'active-cell' : ''}`}
                                    style={{ width: colWidths[col.key], minWidth: col.width }}
                                    onMouseDown={(e) => {
                                        handleMouseDown(e, rIndex, cIndex);
                                    }}
                                    onMouseEnter={() => handleMouseEnter(rIndex, cIndex)}
                                    onDoubleClick={(e) => handleDoubleClick(e, task, col)}
                                >
                                    {content}
                                </div>
                            )
                        })}
                    </div>
                ))}

                <div
                    className="flex items-center justify-center p-2 border-t border-[var(--border-color)] hover:bg-[var(--bg-secondary)] cursor-pointer text-[var(--text-muted)] hover:text-[var(--text-primary)] transition-colors"
                    onClick={onAddRow}
                >
                    <Plus size={16} className="mr-2" /> Add 10 Rows
                </div>
            </div>

            {/* Floating In-Place Editor */}
            {editorState && (
                <FloatingEditor
                    state={editorState}
                    onClose={() => setEditorState(null)}
                    onSave={handleSaveEditor}
                />
            )}

            {contextMenu && (
                <div
                    className="fixed bg-[var(--bg-secondary)] border border-[var(--border-color)] rounded shadow-xl z-50 py-1"
                    style={{ top: contextMenu.y, left: contextMenu.x }}
                >
                    <div className="px-4 py-2 hover:bg-[var(--bg-tertiary)] cursor-pointer text-sm text-[var(--text-primary)]" onClick={handleBulkDeleteRows}>
                        Delete {selectedTasks.length > 1 ? `${selectedTasks.length} rows` : 'Row'}
                    </div>
                    <div className="px-4 py-2 hover:bg-[var(--bg-tertiary)] cursor-pointer text-sm text-[var(--text-primary)]" onClick={() => { onAddRow(); setContextMenu(null); }}>
                        Insert Row Below
                    </div>
                </div>
            )}
        </div>
    );
}

// --- Floating Editor Component ---
const FloatingEditor = ({ state, onClose, onSave }) => {
    const [value, setValue] = useState(state.initialValue);
    const textareaRef = useRef(null);

    // Calculate dimensions
    // Increase width: Cell width + 20% or minimum 300px (to handle small cells)
    // Actually, user just said "increase width a bit". Let's maximize it a bit but keep it reasonable.
    // Google sheets expands to the right. 
    // Let's explicitly set it to be wider, e.g., max(cellWidth + 100, 400) but constrained by viewport?
    // Let's try: width = rect.width + 50.

    // Max height: available space below the cell
    const availableHeight = window.innerHeight - state.rect.top - 20; // 20px padding from bottom

    useLayoutEffect(() => {
        if (textareaRef.current) {
            textareaRef.current.style.height = 'auto';
            // Limit height to available space
            const neededHeight = textareaRef.current.scrollHeight;
            textareaRef.current.style.height = Math.min(neededHeight, availableHeight) + 'px';
        }
    }, [value, availableHeight]);

    const handleBlur = () => {
        onSave(state.taskId, state.field, value, state.isResult);
    };

    const handleKeyDown = (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            handleBlur();
        }
        if (e.key === 'Escape') {
            onClose();
        }
    };

    return (
        <div
            style={{
                position: 'fixed',
                top: state.rect.top - 1,
                left: state.rect.left - 1,
                // Width logic: Minimum of (Cell Width + 50px) or (500px), but at least Cell Width
                width: Math.max(state.rect.width + 60, 400),
                zIndex: 1000,
                maxHeight: 300,
                display: 'flex',
                flexDirection: 'column'
            }}
        >
            <textarea
                ref={textareaRef}
                autoFocus
                value={value}
                onChange={(e) => setValue(e.target.value)}
                onBlur={handleBlur}
                onKeyDown={handleKeyDown}
                className="w-full resize-none p-[8px] text-[0.875rem] leading-[1.25rem] border-2 border-[var(--accent-primary)] rounded-md bg-[var(--bg-primary)] text-[var(--text-primary)] shadow-2xl focus:outline-none custom-scrollbar"
                style={{
                    boxSizing: 'border-box',
                    overflowY: 'auto', // Enable scrolling if content exceeds height
                    maxHeight: availableHeight
                }}
            />
        </div>
    );
};
